require( "scripts/scene_common.lua" );
require("quest/convout.lua");

SLEEP = coroutine.yield;		-- easier to type synonym :: eg use SLEEP(RESUME.ON_WALKTO)
SECOND_DELAY = 30;
MINUTE_DELAY = 1800;

-- converts seconds to milliseconds
function SECONDS(second)
	return second*1000;
end

-- converts minutes to milliseconds
function MINUTES(minute)
	return SECONDS(minute*60);
end


-- Remember to use "combine_speech_tables" to merge the shared dialog with the room dialog
--
shared_dialogtable = dialogtable;
shared_speakertable = speakertable;

function combine_tables(source, dest)
	for k, v in pairs(source) do
		if dest[k] ~= nil then
			-- xprint("OVERLAPPED STRING SCRIPT: ", k);
			GlobalLuaError("Duplicated script in strings:", k);
			return;	-- just give up on the combining
		end;
		dest[k] = v;
	end
end;

function combine_speech_tables()
	-- xprint("combine_speech_tables DIALOG!!");
	combine_tables(shared_dialogtable, dialogtable);
	-- xprint("combine_speech_tables SPEAKER!!");
	combine_tables(shared_speakertable, speakertable);
end;

function lookup_speaker(oldspeaker, line_id)
	if (oldspeaker == "" or oldspeaker == nil) and speakertable then
		oldspeaker = speakertable[line_id];
	end;
	return oldspeaker;
end

function joinTables(source, dest)
	--for k,v in ipairs(t2) do table.insert(t1, v) end return t1   
	for k, v in pairs(source) do
		dest[k] = v;
	end
	return dest;
end

-----------------------------------
-- Debugging output
-----------------------------------
function xprint( ... ) 
  local out = ":::::::::: "; 
  table.foreachi( arg, 
		function(i, value) 
			out = out .. tostring(value) 
		end 
  ); 
  DebugOut( out ); 
end

-----------------------------------
-- Face one actor toward another
-----------------------------------
function actors_face(name_a, name_b)
	-- xprint("actors face[", name_a, "][", name_b, "]");
	scene[name_a]:FaceActor(scene[name_b]);
	scene[name_b]:FaceActor(scene[name_a]);
end

function safe_talk_wait(actor)

	-- AG Modified to not automatically progress conversation at Michelle's request
	local why = SLEEP(RESUME.ON_VOXDONE, RESUME.ON_CLICK);
	--local why = SLEEP(RESUME.ON_CLICK);
	
	if why == RESUME.ON_VOXDONE then
		if actor then actor:StopMouth(); end;
		-- wait for the callback to make SURE it's done
		--SLEEP(RESUME.ON_VOXQUITEDONE, RESUME.ON_CLICK);
		SLEEP(RESUME.ON_CLICK);
	end;
	
end

-----------------------------------
-- Run a multi-line quip
-----------------------------------
function monologue(speaker, monologue_name)
	local was_in_mode = scene[speaker]:GetMode();	-- remember how we started
	
	scene[speaker]:SetMode(MODE.STATIC);
	scene[speaker]:StopWalking();
	
	next_line_id = monologue_name;
	next_index = 1

	while true do			
		-- Create the numbered line id "conv_introduction_1"
		-- xprint("MONOLOGUE[", next_line_id, ":", next_index, "]");
		local cur_line_id = next_line_id .. "_" .. next_index;
	
		-- anticipate the next line		
		next_index = next_index + 1;
	
		if not scene:grabstring(cur_line_id) then
			-- xprint("MONOLOGUE OVER");
			break;	
		end;
		
		-- if the CURRENT line has a handler then run it now
		if dialogtable and dialogtable[cur_line_id] then
			-- xprint("SUPERCONV COMMAND:");
			dialogtable[cur_line_id]();
		end;

		-- say the line and wait
		scene[speaker]:Say(cur_line_id);
		
		safe_talk_wait(scene[speaker]);
	
	end;
	
	scene[speaker]:Say("");
	-- sfx:KillVox();
	scene[speaker]:SetMode(was_in_mode);			-- return to original state
end; -- monologue

-----------------------------------
-- Run a quip (check for multiline)
-----------------------------------
function quip(speaker, stringid)
	scene:SetMode(MODE.DIALOG);
	
	-- xprint("quip(", speaker, ",", stringid, ")");
	if not scene:grabstring(stringid) then
		-- xprint("Referring quip to monologue");
		monologue(speaker, stringid);
	else
		-- xprint("performing quip");
		local actor;
		local line_raw;
		actor, line_raw = scene:grabspeechline(stringid);
		
		if actor == "" or actor == nil then
			line_raw = scene:grabstring(stringid);
		else
			speaker = actor;
		end;

		speaker = lookup_speaker(speaker, stringid);

		-- stop us in our tracks		
		local was_in_mode = scene[speaker]:GetMode();	-- remember how we started
		scene[speaker]:SetMode(MODE.STATIC);
		scene[speaker]:StopWalking();

		-- do the line, kid	
		scene[speaker]:SayRaw(line_raw, -1);
		sfx:PlayVox(stringid, line_raw);

		safe_talk_wait(scene[speaker]);
		
		-- clean up the air over our head
		scene[speaker]:Say("");
		-- sfx:KillVox();
		
		scene[speaker]:SetMode(was_in_mode);			-- return to original state
	end;
	
	scene:SetMode(MODE.FOLLOWCLICKS);
end;


-----------------------------------------------------
-- Fire off a simple conversation from a handler
-- hold the main actor until conv is over
-----------------------------------------------------
function convproc(actorname, conv_base_id)
	scene[actorname]:StartProc(
		function(actor) 
			local was_in_mode = actor:GetMode();	-- remember how we started
			actor:SetMode(MODE.DIALOG);				-- stop us in our tracks
			actor:StopWalking();					-- stop us in our tracks		
			run_super_conversation(actor, conv_base_id);	-- do the chat
			actor:SetMode(was_in_mode);				-- return to original state
		end
	);
end

-----------------------------------------------------
-- Fire off a possibly multiline quip from a handler
-----------------------------------------------------
quipproc = convproc;

function on_misuse_inv(item, flo, screen_loc)
	scene:SetMode(MODE.DIALOG);
	local COMMA = ",";
	xprint( "GENERIC ON_MISUSE_INV:", item, COMMA, flo.name, COMMA, screen_loc.x, COMMA, screen_loc.y );
	
	local currentScene = game:GetCurrentScene();
	
	if currentScene == "m1_petspa" and item == "snookums" then
		flo:StartProc(give_walter_snookums);
	elseif currentScene == "m1_petspa" and item == "dyeremover" then
		convproc("FLO", "inv_dyeremover_cantuse");
	elseif currentScene == "m1_beauty" and item == "dye" then
		convproc("FLO", "quip_give_dye");
	elseif currentScene == "m1_spendalot2" and item == "dogtreat" and game:GetString("map_position") == "m1_park" then
		flo:StartProc(give_snookums_dogtreat);
	elseif currentScene == "m1_dress" and item == "curtains" then
		flo:StartProc(give_bobbie_lace);
	elseif currentScene == "subway" and (item == "token" or item == "gold_token") then
		flo:StartProc(subway_token_misuse);
	elseif currentScene == "m2_sizzle2" and (item == "token" or item == "gold_token") then
		flo:StartProc(bus_token_misuse);
	elseif currentScene == "m3_signatures" and item == "permit" and game:Get( "called_colin" ) > 0 then
		flo:StartProc(no_pen_no_sign);
	else
		-- don't say anything whilst in mini-game mode
		if game:Get("minigame") < 1 then
			local stringid = "misused_" .. item;
			local line_raw = scene:grabstring(stringid);
			
			if line_raw == nil then
				stringid = "misused_inv";
				line_raw = scene:grabstring(stringid);
			end
			
			flo:StopWalking();
			flo:SayRaw(line_raw, -1);
			sfx:PlayVox(stringid, line_raw);
		end
		
		--game:SetString("inventory_fly_back_item", item);
		--game:Set("inventory_fly_back_loc_X", screen_loc.x);
		--game:Set("inventory_fly_back_loc_Y", screen_loc.y);
		--flo:StartProc(inventory_fly_back)
		
		flo:StartProc(on_use_inv_response);

		--game:ClearInvUsage();
		
		-- bug to investigate:
		-- this return value doesn't make it back to C++, if we do the above StartProc.
		-- workaround - C++ will assume the click was handled for now.
		
		-- this prevents the clickhandlers from sending Petra to the site of the misuse.
		-- return RESPONSE.INV_HANDLED;
	end
end

function inventory_fly_back(flo)
	local key = game:GetString("inventory_fly_back_item");

	-- scroll so the item's SLOT is visible
	BAG_INV:PositionInventoryForVisibility(key);
	
	-- go straight there (don't wait)
	BAG_INV:JumpInventoryToTarget(key);
	
	-- disable the item iteself
	BAG_INV:ShowInventory(key, false);
	
	-- do the work
	scene.SETUP:Spawn(inventory_flyback_spec);
	coroutine.yield(RESUME.ON_SIGNAL);
	
	-- re-enable the item
	BAG_INV:ShowInventory(key, true);
	
	delay(30);
	scene.INVENTORY_FLYBACK:Expire();
	
	-- clean up
	game:SetString("inventory_fly_back_item", "");
	game:Set("inventory_fly_back_loc_X", "");
	game:Set("inventory_fly_back_loc_Y", "");
	
	on_use_inv_response(flo);
end

function on_use_inv(item, flo, target, response, flyback)
	scene:SetMode(MODE.DIALOG);
	xprint("GENERIC ON_USE_INV(", item, ",", flo.name, ",", target.name, ")" );
	local inv_cantuse = game:GetInvUsage_CantUse();
	local obj_cantuse = target.cantuse;
	local screen_loc = scene:GetClickedAtScreen();
	local currentScene = game:GetCurrentScene();
	
	-- xprint("inv_cantuse = ", inv_cantuse);
	-- xprint("obj_cantuse = ", obj_cantuse);
	
	if currentScene == "m1_petspa" and item == "snookums" then
		flo:StartProc(give_walter_snookums);
	elseif currentScene == "m1_spendalot2" and item == "dogtreat" and game:GetString("map_position") == "m1_park" then
		flo:StartProc(give_snookums_dogtreat);
	elseif currentScene == "m1_dress" and item == "curtains" then
		flo:StartProc(give_bobbie_lace);
	elseif item == "wirecutters" and game:Get("asked_ethel_for_beads") < 1 then
		flo:StopWalking();
		flo:Say("misused_inv");
		flo:StartProc(on_use_inv_response);
	elseif currentScene == "subway" and (item == "token" or item == "gold_token") then
		flo:StartProc(subway_token_misuse);
	elseif currentScene == "m2_sizzle2" and (item == "token" or item == "gold_token") then
		flo:StartProc(bus_token_misuse);
	else
		
		if inv_cantuse == "" or inv_cantuse == nil then 
			inv_cantuse = "inv_generic_cantuse"; 
			-- xprint("inv_cantuse = ", inv_cantuse);
		end;

		if obj_cantuse == "" or obj_cantuse == nil then 
			obj_cantuse = "obj_generic_cantuse"; 
			-- xprint("obj_cantuse = ", obj_cantuse);
		end;

		
		if response == nil or response == true then
			local line = scene:grabformattedstring(inv_cantuse, { "%"..obj_cantuse.."%" } );
			xprint("result line = ", line);
		
			flo:SayRaw(line, -1);
			sfx:PlayVox(inv_cantuse, line);
		end

		--if flyback == nil or flyback == true then
		--	game:SetString("inventory_fly_back_item", item);
		--	game:Set("inventory_fly_back_loc_X", screen_loc.x);
		--	game:Set("inventory_fly_back_loc_Y", screen_loc.y);
		--	flo:StartProc(inventory_fly_back)
		--else
			flo:StartProc(on_use_inv_response)
		--end
		
		--flo:StartProc(on_use_inv_response)
	end
end

function on_use_inv_response(flo)
 xprint("on_use_inv_response WAITS");
	safe_talk_wait(flo);
	sfx:KillVox();
	flo:SayRaw("");
	scene:SetMode(MODE.FOLLOWCLICKS);
 xprint("on_use_inv_response RESUME");
end

function delay(t)
	coroutine.yield(RESUME.ON_DELAY, t);
end

-----------------------------------
-- simple wait for anim to finish
-----------------------------------
function wait_anim( ... )
	local actor = nil;
	
	table.foreachi( arg, 
		function( i, v )
			if i == 1 then
				actor = v;
			elseif i == 2 then
				GlobalLuaError("Too many arguments in wait_anim function. Expect 0 or 1");
			end
		end
	);
	
	if actor ~= nil then
		coroutine.yield(RESUME.ON_OTHER_ANIM, actor);
	else
		coroutine.yield(RESUME.ON_ANIM);
	end
end

-----------------------------------
-- simple wait for walkto to finish
-----------------------------------
function wait_walkto( ... )
	local actor = nil;
	
	table.foreachi( arg, 
		function( i, v )
			if i == 1 then
				actor = v;
			elseif i == 2 then
				GlobalLuaError("Too many arguments in wait_walkto function. Expect 0 or 1");
			end
		end
	);
	
	if actor ~= nil then
		coroutine.yield(RESUME.ON_OTHER_WALKTO, actor);
	else
		coroutine.yield(RESUME.ON_WALKTO);
	end
end

-----------------------------------
-- simple wait for fade to finish
-----------------------------------
function wait_fade()
	coroutine.yield(RESUME.ON_FADE);
end

function on( ... )
	-- xprint("COMMAND ON");
	table.foreachi( arg, 
		function( i, choicename )
			-- xprint("SUPERCONV>>>>>>>>>on[", choicename, "]");
			gstruct.conv_choices[choicename] = true;
		end
	);
end;

-----------------------------------
-- wait for subscene to finish
-----------------------------------
function wait_subscene()
	delay(2);
end

-----------------------------------
-- Shift an actor some distance along the X axis
-----------------------------------
function move_actor_x(actor, dist)
	if actor then
		local pos = actor:GetPosition();
		pos.x = pos.x + dist; 
		actor:JumpToPoint(pos);
	end
end

function move_actor_y(actor, dist)
	if actor then
		local pos = actor:GetPosition();
		pos.y = pos.y + dist; 
		actor:JumpToPoint(pos);
	end
end

function safe_kill_actor(actor)	
	if actor then
		actor:Expire();
	end
end

function safe_spawn_actor(actor_spec)
	if not scene[actor_spec.name] then
		return scene:Spawn(actor_spec);
	else
		return scene[actor_spec.name];
	end
end

debug_out_spec = 
{
	name = "DEBUG_OUT";
	
	command = function(actor)
		actor:ModifySaySpec
		{
			color = { a = 1, r = 0, g = 1, b = 0 },
			x = 400,
			y = 0,
			w = 400,
			h = 80,
			bubble = false,
			outline_size = 1,
			lineheight = 20,
			fixed = true,
			font = "fonts/garamondpremierpro.mvec",
			anchor = ANCHOR.TOP,
			halign = HALIGN.RIGHT
		};
	end;
}

function debug_out(text)
	if IsCheatMode() then
		if scene.DEBUG_OUT then
			scene.DEBUG_OUT:SayRaw(text);
		end
	end
end

-- can be called at the beginning of every scene
function fade_scene_up(minigame, nofade)
	if IsCheatMode() then
		scene:Spawn(debug_out_spec);
	end
	
	-- metrics for uniquely visited scenes
	if game:Get("MET_visited_scene" .. game:GetCurrentScene()) < 1 then
		game:Set("MET_visited_scene" .. game:GetCurrentScene(), 1);
		MET_Count("UniqueLocationVisited");
	end
	
	scene:ModifySaySpec
	{
		y = 100,	h = 100,
		x = 100,	w = 600,
		lineheight = 20,
		font = "fonts/arial.mvec",
		outline_size = 1,
		color = { a = 1, r = 1, g = 1, b = 1 };
	};
	
	SetToolTipFillColor { a = 0, r = 1, g = 1, b = 0 };
	SetToolTipEdgeColor { a = 0, r = 0, g = 0, b = 0 };
	SetToolTipMargins(10, 2);

	scene:SetTextBoundsXY(20, 20);
	scene:SetMusicVolumeTarget(1);
	
	if nofade == nil or nofade == false then
		scene:SetFadeTarget { a = 0, r = 0, g = 0, b = 0 };
		scene:SetFadeColor { a = 1, r = 0, g = 0, b = 0 };
		scene:SetFadeSpeed(0.05);
		scene:SetMusicVolumeSpeed(0.05);
	else
		scene:SetFadeTarget { a = 0, r = 0, g = 0, b = 0 };
		scene:SetFadeColor { a = 0, r = 0, g = 0, b = 0 };
		scene:SetFadeSpeed(1.0);
		scene:SetMusicVolumeSpeed(1.0);
	end
	
	if IsCheatMode() then
		--game:Set("got_napkin", 1);
		--game:Set("got_map", 1);
	end
	
	if minigame == nil or minigame == false then
		if game:Get("got_napkin") == 1 then
			spawn_button(napkin_button_spec);
		end
		
		if game:Get("got_map") == 1 then
			spawn_button(map_button_spec);
		end
	
		spawn_butterflies();
		spawn_recyclables();
		game:Set("minigame", 0);
	else
		game:Set("minigame", 1);
		spawn_button(back_button_spec);
		spawn_button(pause_button_help_spec);
	end
	
	raise_hud();
end



function hud_raised()
	return GetHUDOffset().y <= 0;
end

function hud_lowered()
	return GetHUDOffset().y >= hud_lower_offset;
end

function raise_hud()
	if not scene.PAUSE_BUTTON then
		spawn_button(pause_button_spec);
	end
	if not hud_raised() then
		if scene.HUD_DROPPER then
			scene.HUD_DROPPER:Expire();
		end
		if not scene.HUD_LIFTER then
			scene:Spawn
			{
				name = "HUD_LIFTER";
				
				command = function(actor)
					local hudOffset = GetHUDOffset();
					while hudOffset.y > 0 do
						hudOffset.y = hudOffset.y - hud_animate_speed;
						SetHUDOffset(hudOffset);
						delay(1);
					end
					
					hudOffset.y = 0;
					SetHUDOffset(hudOffset);
					actor:Expire();
				end;
			}
		end
	end
end

function lower_hud()
	if not hud_lowered() then
		if scene.HUD_LIFTER then
			scene.HUD_LIFTER:Expire();
		end
		if not scene.HUD_DROPPER then
			scene:Spawn
			{
				name = "HUD_DROPPER";
				
				command = function(actor)
					local hudOffset = GetHUDOffset();
					while hudOffset.y < hud_lower_offset do
						hudOffset.y = hudOffset.y + hud_animate_speed;
						SetHUDOffset(hudOffset);
						delay(1);
					end
					
					hudOffset.y = hud_lower_offset;
					SetHUDOffset(hudOffset);
					actor:Expire();
					if scene.PAUSE_BUTTON then
						scene.PAUSE_BUTTON:Expire();
					end
					
					game:ClearInvUsage();
				end;
			}
		end
	end
end

function spawn_help_panel(no_raise)
	while not hud_lowered() do
		delay(1);
	end
	scene:Spawn(help_panel_spec);
	scene:Spawn(help_panel_blink_spec);
	
	scene:Spawn
	{
		name = "HUD_SPAWNER";
		
		command = function(actor)
			if no_raise == nil or no_raise == false then
				local position = hud_lower_offset;
				while position > 0 do
					position = position - hud_animate_speed;
					move_actor_y(scene.HELP_PANEL, -hud_animate_speed);
					move_actor_y(scene.HELP_PANEL_BLINK, -hud_animate_speed);
					move_actor_y(scene.BACK_BUTTON, -hud_animate_speed);
					move_actor_y(scene.PAUSE_BUTTON_HELP, -hud_animate_speed);
					move_actor_y(scene.HINT_BUTTON, -hud_animate_speed);
					move_actor_y(scene.START_BUTTON, -hud_animate_speed);
					move_actor_y(scene.CHECK_DISPLAY_BUTTON, -hud_animate_speed);
					delay(1);
				end
			else
				move_actor_y(scene.HELP_PANEL, -hud_lower_offset);
				move_actor_y(scene.HELP_PANEL_BLINK, -hud_lower_offset);
				move_actor_y(scene.BACK_BUTTON, -hud_lower_offset);
				move_actor_y(scene.PAUSE_BUTTON_HELP, -hud_lower_offset);
				move_actor_y(scene.HINT_BUTTON, -hud_lower_offset);
				move_actor_y(scene.START_BUTTON, -hud_lower_offset);
				move_actor_y(scene.CHECK_DISPLAY_BUTTON, -hud_lower_offset);
			end
		end
	}
end

function kill_help_panel()
	local position = 0;
	while position < hud_lower_offset do
		position = position + hud_animate_speed;
		move_actor_y(scene.HELP_PANEL, hud_animate_speed);
		move_actor_y(scene.HELP_PANEL_BLINK, hud_animate_speed);
		move_actor_y(scene.BACK_BUTTON, hud_animate_speed);
		move_actor_y(scene.PAUSE_BUTTON_HELP , hud_animate_speed);
		move_actor_y(scene.HINT_BUTTON, hud_animate_speed);
		move_actor_y(scene.START_BUTTON, hud_animate_speed);
		move_actor_y(scene.CHECK_DISPLAY_BUTTON, hud_animate_speed);
		move_actor_y(scene.RESET_ACTIVITY_BUTTON, hud_animate_speed);
		delay(1);
	end

	safe_kill_actor(scene.HELP_PANEL);
	safe_kill_actor(scene.HELP_PANEL_BLINK);
	safe_kill_actor(scene.BACK_BUTTON);
	safe_kill_actor(scene.PAUSE_BUTTON_HELP);
	safe_kill_actor(scene.HINT_BUTTON);
	safe_kill_actor(scene.START_BUTTON);
	safe_kill_actor(scene.CHECK_DISPLAY_BUTTON);
	safe_kill_actor(scene.RESET_ACTIVITY_BUTTON);
	--raise_hud();
end

-- function to handle fading to a scene
-- actor starts a new proc to be safe
shared_scene_string = ""; -- hack to pass scene string to actor proc
function goto_scene(actor, scene_string, no_fade)
	if no_fade == nil or no_fade == false then
		scene:SetFadeTarget { a = 1, r = 0, g = 0, b = 0 };
		scene:SetFadeSpeed(0.05);
		scene:SetMusicVolumeSpeed(0.05);
	else
		scene:SetFadeTarget { a = 0, r = 0, g = 0, b = 0 };
		scene:SetFadeSpeed(1.0);
		scene:SetMusicVolumeSpeed(1.0);
	end
	
	shared_scene_string = scene_string;
	actor:StartProc(actor_goto_scene);
end

function actor_goto_scene(actor)
	actor:SetMode(MODE.CUTSCENE);
	wait_fade();
	
	scene:ChainTo(shared_scene_string);
end

function fmod(a, b)
	return a-b*math.floor(a/b);
end

function prepare_actor_text(actor, individual_spec)
	actor:ModifySaySpec(common_text_spec);
	if individual_spec then
		actor:ModifySaySpec(individual_spec);
	end
end

function nav_arrow_point_at_fn(actor, enter)
	if scene.FLO:GetMode() ~= MODE.CUTSCENE then
		if enter then
			actor:SetAlpha(1);
		else
			actor:SetAlpha(0);
		end;
	end
end

function add_quest(quest_name, goto_quests, minigame)
	if not quest_added(quest_name) and not quest_complete(quest_name) then
		game:Set("napkin_" .. quest_name, 1);
	end
	
	if goto_quests == nil or goto_quests == true then
		napkin_button_click(scene.FLO, minigame);
		--scene:ChainToSubScene("napkin");
	end
end

function complete_quest(quest_name, goto_quests, minigame)
	if not quest_complete(quest_name) then
		game:Set("napkin_" .. quest_name, 3);
	end
	if goto_quests == nil or goto_quests == true then
		napkin_button_click(scene.FLO, minigame);
		--scene:ChainToSubScene("napkin");
	end
end

function quest_added(quest_name)
	return game:Get("napkin_" .. quest_name) > 0;
end

function quest_complete(quest_name)
	return game:Get("napkin_" .. quest_name) > 2;
end

function addInventory(desc)
	local key = desc.key;
	
	if not BAG_INV:CheckInventory(key) then
		BAG_INV:AddInventory(desc);
		
		-- scroll so the item's SLOT is visible
		BAG_INV:PositionInventoryForVisibility(key);
		
		-- go straight there (don't wait)
		BAG_INV:JumpInventoryToTarget(key);
		
		-- disable the item iteself
		BAG_INV:ShowInventory(key, false);
		
		-- do the work
		local desiredLocation = BAG_INV:GetInventoryLocation(key); -- { x = 50, y = 550 };
		do_inventory_popup(scene.FLO, key, desiredLocation);
		--scene:Spawn(inventory_popup_spec);
		--coroutine.yield(RESUME.ON_SIGNAL);

		-- re-enable the item
		BAG_INV:ShowInventory(key, true);
		
		-- pause for affect
		delay(50);
	end
end

function addButterflyBag()
	if game:Get("got_butterflybag") ~= 1 then
		game:Set("got_butterflybag", 1);
		
		local desiredLocation = copy_table(butterflybag_spec.position);
		desiredLocation.x = desiredLocation.x + 30;
		desiredLocation.y = desiredLocation.y + 35
		do_inventory_popup(scene.FLO, "butterfly_bag", desiredLocation);
		
		spawn_button(butterflybag_spec);
	end
end

function do_inventory_popup(player, item, desiredLocation)
	local inventory_popup = scene:Spawn
	{
		name = "INVENTORY_POPUP";
		
		command = function(actor)
			actor:SetDepthMode(DEPTH.DEPTH_OVER_HUD)
			actor:AddFlag(AFLAG.AFLAG_FIXED_SCREEN);
		end;
	};
	
	if item ~= "" then
		local popupScale = 1;
		local moveTime = 25;
		local flashFrequency = 20;
		local flashTime = 80;
		local desiredSize = 53; -- inventory size = 53x53
		local desiredPositionFudge = { x = -6, y = 4 };
		local imageStr = "popup/pop_inv_" .. item;
		local imageHaloStr = "popup/pop_inv_" .. item .. "_halo";
		
		-- if we have the popup halo image, load the popup image
		inventory_popup:LoadImage(imageHaloStr);
		if inventory_popup:HasGraphics() then
			inventory_popup:LoadImage(imageStr);
		end
		
		-- if we have the popup image, do the fancy effects
		if inventory_popup:HasGraphics() then
			sfx:PlaySFX("audio/sfx/get_item.ogg");
			inventory_popup:SetScale(popupScale);
			--inventory_popup:AddFlag(AFLAG.AFLAG_NEW_PARTY);
			
			local imageSize = inventory_popup:GetImageSize();
			local popupLocation = { x = 400 - (imageSize.x / 2), y = 300 - (imageSize.y / 2) };
			local desiredScale = desiredSize / (math.max(imageSize.x, imageSize.y));
			
			if hud_lowered() then
				popupLocation.y = popupLocation.y - hud_lower_offset;
			end
			
			inventory_popup:JumpToPoint(popupLocation)
			--if hud_lowered() then
			--	inventory_popup:SetPartyOffset( { x = (imageSize.x / 2), y = (imageSize.y / 2) + hud_lower_offset } );
			--else
			--	inventory_popup:SetPartyOffset( { x = (imageSize.x / 2), y = (imageSize.y / 2) } );
			--end
			--inventory_popup:LoadParty("fx/inventory_party.lua");
			
			scene:Spawn
			{
				name = "INVENTORY_PARTY";
				
				command = function(actor)
					actor:AddFlag(AFLAG.AFLAG_FIXED_SCREEN);
					actor:JumpToPoint{ x = 400, y = 250 };
					actor:LoadPartyContainer("fx/afpfx_starburst_fireworks.lua");
					actor:SetDepthValue(600);
				end
			};
			
			desiredLocation.x = desiredLocation.x - ((imageSize.x / 2) * desiredScale) + desiredPositionFudge.x;
			desiredLocation.y = desiredLocation.y - ((imageSize.y / 2) * desiredScale) + desiredPositionFudge.y;
		
			local timer = 0;
			local halo = true;
			
			-- flash
			while timer < flashTime do
				if halo then
					inventory_popup:LoadImage(imageStr);
					halo = false;
				else
					inventory_popup:LoadImage(imageHaloStr);
					halo = true;
				end
				
				timer = timer + flashFrequency;
				delay(flashFrequency);
			end
			
			timer = 0;
			
			-- move to inventory location
			while timer < moveTime do
				local newPoint = { x = 0, y = 0 };
				local delta = timer / moveTime;
				
				newPoint.x = popupLocation.x + ((desiredLocation.x - popupLocation.x) * delta);
				newPoint.y = popupLocation.y + ((desiredLocation.y - popupLocation.y) * delta);
				inventory_popup:JumpToPoint(newPoint);
				inventory_popup:SetScale(popupScale + ((desiredScale - popupScale) * delta));
				timer = timer + 1;
				delay(1);
			end
			
			inventory_popup:SetAlpha(0);
			
			scene.INVENTORY_PARTY:Expire();
		else
			coroutine.yield(RESUME.NOW);
		end
	end
			
	inventory_popup:Expire();
end

inventory_flyback_spec = 
{
	name = "INVENTORY_FLYBACK";
	
	command = function(actor)
		if item ~= "" then
			local item = game:GetString("inventory_fly_back_item");
			local moveTime = 20;
			local imageSize = { x = 53, y = 53 };
			local desiredPositionFudge = { x = -6, y = 4 };
		
			actor:SetDepthMode(DEPTH.DEPTH_OVER_HUD)
			actor:AddFlag(AFLAG.AFLAG_FIXED_SCREEN);
			actor:LoadImage("inv/inv_" .. item);
			actor:AddFlag(AFLAG.AFLAG_NEW_PARTY);
			
			local inv_cursor_hot = BAG_INV:GetInventoryCursorHot(item);
			local popupLocation = { x = game:Get("inventory_fly_back_loc_X") - inv_cursor_hot.x, y = game:Get("inventory_fly_back_loc_Y") - inv_cursor_hot.y };
			--local popupLocation = { x = 400 - (imageSize.x / 2), y = 300 - (imageSize.y / 2) };
			
			actor:JumpToPoint(popupLocation)
			actor:SetPartyOffset( { x = (imageSize.x / 2), y = (imageSize.y / 2) } );
			actor:LoadParty("fx/inventory_party_small.lua");
			
			local desiredLocation = BAG_INV:GetInventoryLocation(item); -- { x = 50, y = 550 };
			desiredLocation.x = desiredLocation.x - (imageSize.x / 2);
			desiredLocation.y = desiredLocation.y - (imageSize.y / 2);
			
			local timer = 0;
			
			-- move to inventory location
			while timer < moveTime do
				local newPoint = { x = 0, y = 0 };
				local delta = timer / moveTime;
				
				newPoint.x = popupLocation.x + ((desiredLocation.x - popupLocation.x) * delta);
				newPoint.y = popupLocation.y + ((desiredLocation.y - popupLocation.y) * delta);
				actor:JumpToPoint(newPoint);
				timer = timer + 1;
				delay(1);
			end
		end
		
		actor:SetAlpha(0);
		scene.FLO:Signal();
	end;
};

function spawn_button(button_spec)	
	return scene:Spawn
	{
		name = button_spec.name;
		halo = "quest/objects/button";
		
		command = function(actor)
			if button_spec.depth_mode then
				actor:SetDepthMode(button_spec.depth_mode);
			end
			actor:AddFlag(AFLAG.AFLAG_FIXED_SCREEN);
			actor:AddFlag(AFLAG.AFLAG_IGNORE_GROUND_MESH);
			actor:AddFlag(AFLAG.AFLAG_NO_VOX);
			actor:SetWalkSpeed(0);
			if button_spec.flags then
				actor:AddFlag(button_spec.flags)
			end
		
			if button_spec.position then
				actor.position = button_spec.position;
			else
				actor.position = { x = 0, y = 0 };
			end
			
			if button_spec.text then
				actor.text = button_spec.text;
			else
				actor.text = "back"
			end
			
			if button_spec.halo then
				actor:LoadZone(button_spec.halo);
			end
			
			if button_spec.depth then
				actor:SetDepthValue(button_spec.depth);
			end
			
			actor:JumpToPoint(actor.position);
			if button_spec.say_spec then
				actor:ModifySaySpec(button_spec.say_spec);
			else
				actor:ModifySaySpec
				{
					color = { a = 1, r = 1, g = 1, b = 1 },
					y = 19,
					x = -10,
					w = 90,
					h = 25,
					bubble = false,
					outline_size = 0,
					font = "fonts/arial.mvec"; 
					anchor = ANCHOR.CENTER
				};
			end
			
			if text ~= "" then
				actor:Say(button_spec.text);
			end
			
			--actor:SayRaw(actor.name);
			--actor:SetDepthMode(DEPTH.DEPTH_OVER_HUD);
			actor.clickfn = back_button_on_click;
			actor:SetPointAtCursor("cursor");
			actor.clickfn = button_spec.on_click;
			actor.buttondown = false;
			
			if button_spec.image then
				actor:LoadImage(button_spec.image .. "_up");
			else
				actor:LoadImage("buttons/button_help_panel");
			end
			
			if button_spec.init then
				button_spec.init(actor);
			end;
		end;
		
		on_click = function(flo, actor)
			if button_spec.image then
				actor:LoadImage(button_spec.image .. "_down");
			else
				actor:LoadImage("buttons/button_help_panel");
			end
			if button_spec.clicksound then
				sfx:PlaySFX(button_spec.clicksound);
			end
			actor.buttondown = true;
			return RESPONSE.IGNORE;
		end;
		
		on_release = function(flo, actor)
			if button_spec.image then
				actor:LoadImage(button_spec.image .. "_up");
			else
				actor:LoadImage("buttons/button_help_panel");
			end
			if actor.clickfn and actor.buttondown == true then
				actor.clickfn(flo, actor);
			end
			actor.buttondown = false;
			return RESPONSE.IGNORE;
		end;
		
		point_at_fn = function(actor, enter)
			if enter then
				if button_spec.image then
					actor:LoadImage(button_spec.image .. "_over");
				else
					actor:LoadImage("buttons/button_help_panel");
				end
				
				if button_spec.text ~= "" then
					if button_spec.say_spec and button_spec.say_spec.color_over then
						actor:ModifySaySpec
						{
							color = button_spec.say_spec.color_over;
						}
					else
						actor:ModifySaySpec
						{
							color = { a = 1, r = 1, g = 1, b = 0 },
						}
					end
					actor:Say(button_spec.text);
				end
				
				if button_spec.rolloversound then
					sfx:PlaySFX(button_spec.rolloversound);
				end
			else
				if button_spec.image then
					actor:LoadImage(button_spec.image .. "_up");
				else
					actor:LoadImage("buttons/button_help_panel");
				end
				if button_spec.text ~= "" then
					if button_spec.say_spec and button_spec.say_spec.color then
						actor:ModifySaySpec
						{
							color = button_spec.say_spec.color;
						}
					else
						actor:ModifySaySpec
						{
							color = { a = 1, r = 1, g = 1, b = 1 },
						}
					end
					actor:Say(button_spec.text);
				end
			end;
		end;
	}
end;

function spawn_nav_arrow(nav_arrow_spec)
	scene:Spawn
	{
		name = nav_arrow_spec.name;
		halo = nav_arrow_spec.halo;
		
		command = function(actor)
			actor:SetPointAtCursor(nav_arrow_spec.cursor);
			
			if nav_arrow_spec.depth then
				actor:SetDepthValue(nav_arrow_spec.depth);
			end;
			
			actor.on_arrive = function(flo, actor)
				flo:StartProc(nav_arrow_spec.goto_fn);
			end;
			
			actor.on_use_inv = function(item, flo, actor)
				game:ClearInvUsage();
				flo:StartProc(nav_arrow_spec.goto_fn);
				return RESPONSE.INV_HANDLED;
			end;
			
			actor.point_at_fn = function(actor, enter)
				if nav_arrow_spec.point_at_fn then
					nav_arrow_spec.point_at_fn(actor, enter);
				end
			end;
		end;
	};
end;

function spawn_simple_hotspot(name, depth, num)
	scene:Spawn
	{
		name = string.upper(name);
		halo = name;
		
		command = function(actor)
			actor:SetDepthValue(depth);
		end;
		
		on_arrive = function(flo, actor)
			flo:StartProc(
				function(flo)
					flo:SetMode(MODE.CUTSCENE);
					if num == nil or num < 2 then
						run_super_conversation(flo, "quip_" .. name);
					else
						conv_multiple(flo, "quip_" .. name .. "_", "a", num);
					end
					flo:SetMode(MODE.FOLLOWCLICKS);
				end
			);
		end;
	}
end

-- degrees to radians - accepts degrees, returns radians
function deg_to_rad(degrees)
	return (math.pi*degrees)/180;
end

function copy_table(table_to_copy)
	local copied_table = {};

	for i, v in table_to_copy do
		copied_table[i] = v;
	end

	return copied_table;
end

function deep_copy_table(object)
	local lookup_table = {}
	local function _copy(object)
		if type(object) ~= "table" then
			return object
		elseif lookup_table[object] then
			return lookup_table[object]
		end
		
		local new_table = {}
		lookup_table[object] = new_table
		for index, value in pairs(object) do
			new_table[_copy(index)] = _copy(value)
		end
		return setmetatable(new_table, getmetatable(object))
	end
	return _copy(object)
end

-- function to handle conversations with sequential outputs on subsequent calls
function conv_multiple(actor, convstub, char, num)
	local convstubvar = game:Get(convstub);
	
	if convstubvar == nil then
		convstubvar = 0;
	end
	
	run_super_conversation(actor, convstub .. string.char(string.byte(char) + convstubvar));
	game:Set(convstub, fmod(convstubvar + 1, num));
end

function spawn_activity_complete_text(text, sound)
	scene:Spawn
	{
		name = "ACTIVITY_COMPLETE";
		
		command = function(actor)
			actor:ModifySaySpec
			{
				bubble = false,
				fixed = true,
				color = { a = 1, r = 1, g = 0, b = 0},
				lineheight = 50,
				anchor = ANCHOR.TOP,
				halign = HALIGN.CENTER,
				font = "fonts/dom_casual_std.mvec",
				outline_size = 1,
				outline_color = { a = 1, r = 0, g = 0, b = 0 },
				w = 2400
			};
			actor:AddFlag(AFLAG.AFLAG_TEXT_ON_ACTOR);
			actor:AddFlag(AFLAG.AFLAG_FIXED_SCREEN);
			actor:AddFlag(AFLAG.AFLAG_NO_VOX);
			actor:JumpToPoint{ x = 400, y = 300 };
			actor:LoadPartyContainer("fx/afpfx_explosion_fireworks.lua");
			actor:ModifySaySpec { x = -1200, y = -100 };
			
			local end_size = 70;
			local start_size = 300;
			local inc = 10;
			local cur_size = start_size;
			
			actor:ModifySaySpec { h = start_size };
			actor:AltTextToImage();
			
			while cur_size > end_size do
				actor:ModifySaySpec { lineheight = cur_size, color = { a = 1-((cur_size-end_size)/(start_size-end_size)), r = 1, g = 0, b = 0 } };
				actor:Say(text);
				cur_size = cur_size - inc;
				delay(1);
			end
			
			sfx:PlaySFX(sound);
			
			local flash_amount = 4;
			local flash_delay = 5;
			local flash_count = 0;
			
			while flash_count < flash_amount do
				actor:ModifySaySpec { color = { a = 0, r = 1, g = 0, b = 0 }, outline_size = 0 };
				actor:Say(text);
				delay(flash_delay);
				actor:ModifySaySpec { color = { a = 1, r = 1, g = 0, b = 0 }, outline_size = 1 };
				actor:Say(text);
				delay(flash_delay);
				flash_count = flash_count + 1;
			end
			
			coroutine.yield(RESUME.ON_CLICK, RESUME.ON_DELAY, 60);
			
			--actor:AltTextToImage();
			scene.FLO:Signal();
			actor:Expire();
		end;
	}
end

function pulse_actor(actor)
	local pulsetime = 30;
	local pulserange = 0.5;
	local count = 0;
	while true do
		while count < pulsetime do
			local tint = pulserange + ((pulsetime-count) / pulsetime)*(1-pulserange);
			actor:SetTint { a = 1, r = tint, g = tint, b = 1 };
			count = count + 1;
			delay(1);
		end
		count = 0;
		while count < pulsetime do
			local tint = pulserange + (count / pulsetime)*(1-pulserange);
			actor:SetTint { a = 1, r = tint, g = tint, b = 1 };
			count = count + 1;
			delay(1);
		end
		count = 0;
	end
end

function flo_after_napkin_prompt(flo)
	local module_progress = game:Get("module_progress");
	local napkin_module_progress = game:Get("napkin_module_progress");
	--debug_out("test " .. module_progress " " .. napkin_module_progress);
	if module_progress <= 1 and napkin_module_progress == 2 then
		
		if BAG_INV:CheckInventory("token") == false then
			run_super_conversation(flo, "quip_flo_general_L");
		else
			run_super_conversation(flo, "conv_goto_module2");
		end
		return true;
	elseif module_progress == 2 and napkin_module_progress == 3 then
		run_super_conversation(flo, "quip_flo_general2_J");
		return true
	elseif module_progress == 3 and napkin_module_progress == 4 then
		run_super_conversation(flo, "quip_flo_general3_G");
		return true;
	else
		return false;
	end
end

function subway_token_misuse(flo)
	flo:SetMode(MODE.CUTSCENE);
	run_super_conversation(flo, "inv_token_cantuse_b");
	flo:SetMode(MODE.FOLLOWCLICKS);
end

function bus_token_misuse(flo)
	flo:SetMode(MODE.CUTSCENE);
	if game:Get("bus_fixed") < 0 then
		run_super_conversation(flo, "quip_bus_flat");
	else
		run_super_conversation(flo, "inv_token_cantuse_c");
	end
	flo:SetMode(MODE.FOLLOWCLICKS);
end

function spawn_bg_actor(_image, depth)
	return scene:Spawn
	{
		name = "BG_ACTOR_" .. _image;
		gfx = { image = _image };
		
		command = function(actor)
			actor:SetDepthValue(depth);
		end;
	};
end

function spawn_load_bar(_name, text, load_total)
	scene:Spawn
	{
		name = _name;
		gfx = { image = "quest/loadbar/dialog" };
		
		command = function(actor)
			actor:SetDepthMode(DEPTH.DEPTH_OVER_HUD);
			actor:SetDepthValue(1);
			actor.load_total = load_total;
			
			actor:ModifySaySpec
			{
				color = { a = 1, r = 0, g = 0, b = 0 },
				y = 170,
				x = 300,
				w = 200,
				h = 200,
				bubble = false,
				lineheight = 30,
				outline_size = 0,
				fixed = true,
				font = "fonts/franklingothic_demicond.mvec"; 
				anchor = ANCHOR.CENTER;
				halign = HALIGN.CENTER;
			};
			
			actor:Say(text);
		end
	}
	
	scene:Spawn
	{
		name = _name .. "_BAR";
		gfx = { image = "quest/loadbar/bar" };
		
		command = function(actor)
			actor:SetDepthMode(DEPTH.DEPTH_OVER_HUD);
			actor:SetDepthValue(2);
			actor:JumpToPoint{x = 284, y = 296};
			actor:SetScaleX(0);
		end
	}
end

function update_load_bar(name, amount)
	local percent = amount / scene[name].load_total;
	
	scene[name .. "_BAR"]:SetScaleX(percent);
	coroutine.yield(RESUME.NOW);
end

function kill_load_bar(name)
	scene[name]:Expire();
	scene[name .. "_BAR"]:Expire();
end

require( "quest/superconv.lua" );
require( "quest/scenelist.lua" );
require( "quest/butterflies/butterflies.lua" );
require( "quest/recyclables/recyclables.lua" );

SetCaptionColorIn { r = 0.85, g = 0.81, b = 0.98 };
SetCaptionColorEdge { r = 0, g = 0, b = 0 };
